# rirt.R
#
# Copyright (C) 2010 Stephane Germain
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

fitirt <- function(data, model="2PLM", key=NULL, graded=FALSE, options.weights=NULL,
  max.em.iter=100, max.nr.iter=200, precision=1e-5, smooth.factor=NULL,
  z=seq(-4, 4, length=64), grouping=TRUE, init=NULL,
  verbose=0, continue.on.error=1,
  slope.init=1.702, thresh.init=0, asymp.init=0.2,
  slope.prior=NULL, thresh.prior=NULL, asymp.prior=NULL)
{
  nbr.subject <- dim(data)[1]
  nbr.item <- dim(data)[2]
  subject <- rownames(data)
  item <- colnames(data)
  if(is.null(subject)) subject <- 1:nbr.subject
  if(is.null(item)) item <- 1:nbr.item

  # match the model name
  if(is.null(model)) stop("invalid model")
  choices <- c("1PLM", "2PLM", "3PLM", "NOMINAL", "GRADED", "PMMLE", "KERNEL")
  model.num <- pmatch(toupper(model), choices, duplicates.ok=TRUE)
  if(any(is.na(model.num))) stop("invalid model")
  model <- choices[model.num]

  if(!is.null(init) && !is.null(init$z)) z <- init$z
  nbr.quad <- length(z)
  quad.from <- z[1]
  quad.to <- z[nbr.quad]

  # defaults values
  kernel <- FALSE
  penalized <- FALSE
  polytomous <- FALSE
  parametric <- TRUE
  method <- "MMLE"
  mixed <- FALSE

  if(length(model) > 1) {
    mixed <- TRUE
    method.num <- rep(0, nbr.item)
    polytomous <- TRUE
    slope.mean <- 0
    slope.dev <- 0
    slope.prior <- 0
    thresh.mean <- 0
    thresh.dev <- 0
    thresh.prior <- 0
    asymp.mean <- 0
    asymp.weight <- 0
    asymp.prior <- 0
  } else if(model %in% c("1PLM", "2PLM", "3PLM")) {
    if(graded) stop("model is incompatible with graded option")
    if(!is.null(key)) stop("model is incompatible with key option")
    if(!is.null(slope.prior) || !is.null(thresh.prior) || !is.null(asymp.prior))
      method <- "BME"
    if(!is.null(slope.prior)) {
      slope.mean <- slope.prior[1]
      slope.dev <- slope.prior[2]
      slope.prior <- 1
    } else {
      slope.mean <- 0
      slope.dev <- 0
      slope.prior <- 0
    }
    if(!is.null(thresh.prior)) {
      thresh.mean <- thresh.prior[1]
      thresh.dev <- thresh.prior[2]
      thresh.prior <- 1
    } else {
      thresh.mean <- 0
      thresh.dev <- 0
      thresh.prior <- 0
    }
    if(!is.null(asymp.prior)) {
      asymp.mean <- asymp.prior[1]
      asymp.weight <- asymp.prior[2]
      asymp.prior <- 1
    } else {
      asymp.mean <- 0
      asymp.weight <- 0
      asymp.prior <- 0
    }

  } else if(model %in% c("PMMLE", "KERNEL")) {
    parametric <- FALSE
    if(graded || !is.null(key)) polytomous <- TRUE
    method <- model
    if(method=="KERNEL") {
      kernel <- TRUE
      if(is.null(smooth.factor)) smooth.factor <- 2.7*nbr.subject^-0.2
    } else if(method=="PMMLE") {
      penalized <- TRUE
      if(is.null(smooth.factor)) smooth.factor <- 4*nbr.subject^0.2
      if(2^floor(log2(nbr.quad)) != nbr.quad)
        stop("nbr.quad has to be a power of 2")
    }
    slope.mean <- 0
    slope.dev <- 0
    slope.prior <- 0
    thresh.mean <- 0
    thresh.dev <- 0
    thresh.prior <- 0
    asymp.mean <- 0
    asymp.weight <- 0
    asymp.prior <- 0

  } else {
    polytomous <- TRUE
    slope.mean <- 0
    slope.dev <- 0
    slope.prior <- 0
    thresh.mean <- 0
    thresh.dev <- 0
    thresh.prior <- 0
    asymp.mean <- 0
    asymp.weight <- 0
    asymp.prior <- 0      
  }

  if(!polytomous) {

    # check the validity of the data
    if(length(data[data!=0 & data !=1 & !is.na(data)])>0)
      stop("the data is not all binary")

    nbr.option.tot <- nbr.item
    items.pos <- 0:(nbr.item-1)
    nbr.options <- rep(1, nbr.item)
    option <- rep(1, nbr.item)
    options.label <- as.list(option)
    options.weights <- rep(1, nbr.item)

  } else {

    # extract the list of options
    options.label <- do.call("c", apply(data, 2, 
      function(x)list(sort(unique(x)))))
    nbr.options <- sapply(options.label, length)

    if(min(nbr.options)<2)
      stop("the number of options to item(s) ",
        paste(item[nbr.options<2], collapse=", ")," is too small")

    if(max(nbr.options)>50)
      stop("the number of options to item(s) ",
        paste(item[nbr.options>50], collapse=", ")," is too large")

    nbr.option.tot <- sum(nbr.options)
    items.pos <- c(0, cumsum(nbr.options[-nbr.item]))

    # transform the options of each item to integer values starting at 1
    data <- sapply(1:nbr.item, function(j)match(data[,j], options.label[[j]]))

    # transform the key
    if(!is.null(key)){
      if(!is.null(dim(key))) key <- apply(key,2,as.vector)
      if(length(key)!=nbr.item) stop("the key is not of the right length")
      names(key) <- item
      key.num <- sapply(1:nbr.item, function(j)match(key[j], options.label[[j]]))
      key.num[mixed & model == "GRADED"] <- 0
      if(any(is.na(key.num)))
        stop("no subject got the right answer to item(s) ",
          paste(item[is.na(key.num)], collapse=", "))
    }

    # construct the weights vector
    if(mixed && !is.null(key)){
      options.weights <- c(sapply(1:nbr.item,
        function(j)(if(model[j]=="GRADED"){1:nbr.options[j]}else{rep(0,nbr.option.tot)})),recursive=TRUE)
      options.weights[(key.num+items.pos)[model=="NOMINAL"]] <- 1
    }else if(graded && is.null(key) && is.null(options.weights)){
      options.weights <- c(sapply(1:nbr.item,function(j)(1:nbr.options[j])),recursive=TRUE)
    }else if(!is.null(key) && !graded && is.null(options.weights)){
      options.weights <- rep(0, nbr.option.tot)
      options.weights[key.num+items.pos] <- 1
    }else if(!is.null(options.weights) && is.null(key) && !graded){
      if(is.list(options.weights)) options.weights <- c(options.weights, recursive=TRUE)
      if(length(options.weights)!=nbr.option.tot) stop("the options.weights is not of the right length")
    }else{
      stop("one and only one of these argument is needed: graded, key or options.weights")
    }

    # now that the weights are done the nominal model is not considered graded
    if(!mixed && any(model == "NOMINAL")) graded <- FALSE

    item <- rep(item, nbr.options)
    option <- do.call("c", options.label)
  }

  # replace the NA by -1
  data[is.na(data)] <- -1

  # initialize
  probs <- rep(0.0, nbr.quad*nbr.option.tot)
  if(!polytomous) {
    slopes <- rep(slope.init, nbr.item)
    thresh <- rep(thresh.init, nbr.item)
    if(model == "3PLM") asymp <- rep(asymp.init, nbr.item)
    else asymp <- rep(0, nbr.item)
  } else {
    thresh <- rep(0, nbr.option.tot)
    if(mixed || any(model != "GRADED")) slopes <- rep(0, nbr.option.tot)
    else slopes <- rep(0, nbr.item)
  }
  initialized <- FALSE
  if(parametric) {
    if(!is.null(init)){
      if(dim(init)[1]!=nbr.option.tot)
        stop("the number of row in init is incorrect")
      slopes <- init$a
      thresh <- init$b
      if(model == "3PLM") asymp <- init$c
      else asymp <- rep(0, nbr.option.tot)
      if(!mixed && any(model == "GRADED")) slopes <- slopes[items.pos+1]
      initialized <- TRUE
    }
  } else {
    if(!is.null(init)){
      probs <- as.matrix(init[,-1])
      initialized <- TRUE
      if(dim(probs)[2]!=nbr.option.tot)
        stop("the number of column in init is incorrect.")
    }
  }

  .C("set_verbose", as.integer(verbose),
    PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)

  .C("continue_on_error", as.integer(continue.on.error),
    PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)

  # call the function from the library
  if(!polytomous) {

    out <- .C("irt_wrapper",
      nbr.subject=as.integer(nbr.subject), nbr.item=as.integer(nbr.item),
      data=as.integer(t(data)), nbr.quad=as.integer(nbr.quad),
      quad.from=as.double(quad.from), quad.to=as.double(quad.to),
      model=as.integer(model.num), penalized=as.integer(penalized), kernel=as.integer(kernel),
      smooth.factor=as.double(smooth.factor), slope.prior=as.integer(slope.prior),
      thresh.prior=as.integer(thresh.prior), asymp.prior=as.integer(asymp.prior),
      slope.mean=as.double(slope.mean), slope.dev=as.double(slope.dev),
      thresh.mean=as.double(thresh.mean), thresh.dev=as.double(thresh.dev),
      asymp.mean=as.double(asymp.mean), asymp.weight=as.double(asymp.weight),    
      max.em.iter=as.integer(max.em.iter), max.nr.iter=as.integer(max.nr.iter),
      precision=as.double(precision), grouping=as.integer(grouping), 
      slopes=as.double(slopes),
      thresh=as.double(thresh),
      asymp=as.double(asymp),
      slopes.stddev=as.double(rep(0.0, nbr.item)),
      thresh.stddev=as.double(rep(0.0, nbr.item)),
      asymp.stddev=as.double(rep(0.0, nbr.item)),
      quad.points=as.double(rep(0.0, nbr.quad)),
      quad.weights=as.double(rep(0.0, nbr.quad)),
      probs=as.double(probs),
      probs.stddev=as.double(rep(0.0, nbr.quad*nbr.item)),
      nbr.notconverge=as.integer(0),
      notconverge=as.integer(rep(0, nbr.item)),
      nbr.ignore=as.integer(0),
      ignore=as.integer(rep(0, nbr.item)),
      initialized=as.integer(initialized),
      adjust.weights=as.integer(0),
      em.converge=as.integer(1),
      PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)

  } else if(mixed) {

    out <- .C("mixed_wrapper",
      nbr.subject=as.integer(nbr.subject),
      nbr.item=as.integer(nbr.item),
      nbr.option.tot=as.integer(nbr.option.tot),
      items.models=as.integer(model.num), 
      items.methods=as.integer(method.num), 
      items.pos=as.integer(items.pos), 
      nbr.options=as.integer(nbr.options),
      data=as.integer(t(data)), 
      options.weights=as.double(options.weights),
      nbr.quad=as.integer(nbr.quad), 
      quad.from=as.double(quad.from),
      quad.to=as.double(quad.to),
      smooth.factor=as.double(smooth.factor),
      max.em.iter=as.integer(max.em.iter), 
      max.nr.iter=as.integer(max.nr.iter),
      precision=as.double(precision), 
      grouping=as.integer(grouping), 
      slopes=as.double(slopes),
      thresh=as.double(thresh),
      slopes.stddev=as.double(rep(0.0, length(slopes))),
      thresh.stddev=as.double(rep(0.0, nbr.option.tot)),
      slope.init=as.double(slope.init),
      thresh.init=as.double(thresh.init),
      quad.points=as.double(rep(0.0, nbr.quad)),
      quad.weights=as.double(rep(0.0, nbr.quad)),
      probs=as.double(probs),
      probs.stddev=as.double(rep(0.0, nbr.quad*nbr.option.tot)),
      iccs=as.double(rep(0.0, nbr.quad*(nbr.item+1))),
      iccs.stddev=as.double(rep(0.0, nbr.quad*(nbr.item+1))),
      nbr.notconverge=as.integer(0),
      notconverge=as.integer(rep(0, nbr.item)),
      nbr.ignore=as.integer(0),
      ignore=as.integer(rep(0, nbr.item)),
      initialized=as.integer(initialized),
      data.exp=as.integer(rep(0,nbr.subject*nbr.option.tot)),
      em.converge=as.integer(1),
      PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)

  } else { # polytomous

    out <- .C("mirt_wrapper",
      nbr.subject=as.integer(nbr.subject),
      nbr.item=as.integer(nbr.item),
      nbr.option.tot=as.integer(nbr.option.tot),
      items.pos=as.integer(items.pos), 
      nbr.options=as.integer(nbr.options),
      data=as.integer(t(data)), 
      options.weights=as.double(options.weights),
      nbr.quad=as.integer(nbr.quad), 
      quad.from=as.double(quad.from),
      quad.to=as.double(quad.to),
      penalized=as.integer(penalized),
      kernel=as.integer(kernel),
      smooth.factor=as.double(smooth.factor),
      graded=as.integer(graded),
      max.em.iter=as.integer(max.em.iter), 
      max.nr.iter=as.integer(max.nr.iter),
      precision=as.double(precision), 
      grouping=as.integer(grouping), 
      slopes=as.double(slopes),
      thresh=as.double(thresh),
      slopes.stddev=as.double(rep(0.0, length(slopes))),
      thresh.stddev=as.double(rep(0.0, nbr.option.tot)),
      slope.init=as.double(slope.init),
      thresh.init=as.double(thresh.init),
      quad.points=as.double(rep(0.0, nbr.quad)),
      quad.weights=as.double(rep(0.0, nbr.quad)),
      probs=as.double(probs),
      probs.stddev=as.double(rep(0.0, nbr.quad*nbr.option.tot)),
      iccs=as.double(rep(0.0, nbr.quad*(nbr.item+1))),
      iccs.stddev=as.double(rep(0.0, nbr.quad*(nbr.item+1))),
      nbr.notconverge=as.integer(0),
      notconverge=as.integer(rep(0, nbr.item)),
      nbr.ignore=as.integer(0),
      ignore=as.integer(rep(0, nbr.item)),
      initialized=as.integer(initialized),
      data.exp=as.integer(rep(0,nbr.subject*nbr.option.tot)),
      adjust.weights=as.integer(0),
      em.converge=as.integer(1),
      PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)

  }

  if(out$nbr.ignore){ 
    warning("theses items were ignored by the estimation process: ",
      paste((1:nbr.item)[out$ignore!=0], collapse=", ")) }
  if(out$nbr.notconverge){ 
    warning("the Newton algorithm didn't converged for these items: ",
      paste((1:nbr.item)[out$notconverge!=0], collapse=", ")) }
  if(!out$em.converge) warning("the EM algorithm didn't converged")

  if(parametric && !polytomous) {
    object <- data.frame(row.names=1:length(item), item=item,
      a=out$slopes, b=out$thresh)
    if(model.num == 3) object$c <- out$asymp

    se <- data.frame(row.names=1:length(item), item=item,
      a.se=out$slopes.stddev, b.se=out$thresh.stddev)
    if(model.num == 3) se$c.se <- out$asymp.stddev

    object$intercept <- -out$slopes*out$thresh

  } else if(parametric && polytomous) {
    if(!mixed && graded) out$slopes <- rep(out$slopes, nbr.options) 
    object <- data.frame(row.names=1:length(item), item=item, option=option,
      a=out$slopes, b=out$thresh)
    if(mixed){
      object$intercept <- rep(NA, nbr.option.tot)
      for(j in 1:nbr.item){
        if(model[j]=="GRADED"){
          bound <- out$thresh[items.pos[j]+1]
          for(k in 1:(nbr.options[j]-1)){
            bound <- 2*out$thresh[items.pos[j]+k]-bound
            object$intercept[items.pos[j]+k] <- -out$slopes[items.pos[j]+k]*bound
          }
        } else {
          for(k in 1:nbr.options[j]){
            object$intercept[items.pos[j]+k] <- -out$slopes[items.pos[j]+k]*out$thresh[items.pos[j]+k]
          }
        }
      }
    }else if(graded){
      object$intercept <- rep(NA, nbr.option.tot)
      for(j in 1:nbr.item){
        bound <- out$thresh[items.pos[j]+1]
        for(k in 1:(nbr.options[j]-1)){
          bound <- 2*out$thresh[items.pos[j]+k]-bound
          object$intercept[items.pos[j]+k] <- -out$slopes[items.pos[j]+k]*bound
        }
      }
    } else {
      object$intercept <- -out$slopes*out$thresh
    }

    if(!mixed && graded) out$slopes.stddev <- rep(out$slopes.stddev, nbr.options)
    se <- data.frame(row.names=1:length(item), item=item, option=option,
      a.se=out$slopes.stddev,  b.se=out$thresh.stddev)

  } else if(!parametric && !polytomous) {
    object <- data.frame(out$quad.points, matrix(out$probs, ncol=nbr.item))
    colnames(object) <- c("z", paste("ICC", item, sep="."))
    
    se <- data.frame(out$quad.points, matrix(out$probs.stddev, ncol=nbr.item))
    colnames(se) <- c("z", paste("se", item, sep="."))
    
  } else if(!parametric && polytomous) {
    object <- data.frame(out$quad.points, matrix(out$probs, ncol=nbr.option.tot))
    colnames(object) <- c("z", paste("OCC", item, option, sep="."))
    
    se <- data.frame(out$quad.points, matrix(out$probs.stddev, ncol=nbr.option.tot))
    colnames(se) <- c("z", paste("se", item, option, sep="."))
    
  }

  attributes(object)$polytomous <- polytomous
  attributes(object)$graded <- graded
  attributes(object)$mixed <- mixed
  attributes(object)$key <- key
  attributes(object)$parametric <- parametric
  attributes(object)$model <- model
  attributes(object)$model.num <- model.num
  attributes(object)$nbr.item <- nbr.item
  attributes(object)$item <- item
  attributes(object)$option <- option
  attributes(object)$nbr.options <- nbr.options
  attributes(object)$nbr.option.tot <- nbr.option.tot
  attributes(object)$items.pos <- items.pos
  attributes(object)$options.label <- options.label
  attributes(object)$options.weights <- options.weights
  attributes(object)$call <- match.call()
  attributes(object)$method <- method
  attributes(object)$se <- se
  attributes(object)$smooth.factor <- smooth.factor
  attributes(object)$nbr.subject <- nbr.subject
  attributes(object)$slope.prior <- slope.prior
  attributes(object)$slope.mean <- slope.mean
  attributes(object)$slope.dev <- slope.dev
  attributes(object)$thresh.prior <- thresh.prior
  attributes(object)$thresh.mean <- thresh.mean
  attributes(object)$thresh.dev <- thresh.dev
  attributes(object)$asymp.prior <- asymp.prior
  attributes(object)$asymp.mean <- asymp.mean
  attributes(object)$asymp.weight <- asymp.weight

  class(object) <- c("rirt", class(object))
  return(object)
}

genirt <- function(nbr.item, model="2PLM", nbr.options=rep(2, nbr.item),
  genslope=function(n)runif(n,0.5,2), genthresh=function(n)runif(n,-2,2),
  genasymp=function(n)runif(n,0,0.2), genslopenom=function(n)runif(n,-2,2),
  fixed.slope=1, items.label=NULL, options.label=NULL, key=NULL)
{

  choices <- c("1PLM", "2PLM", "3PLM", "NOMINAL", "GRADED")
  model <- pmatch(toupper(model), choices, duplicates.ok=TRUE)
  if(any(is.na(model))) stop("invalid model")
  model <- choices[model]

  if(length(nbr.item)!=1 || nbr.item!=round(nbr.item) || nbr.item<1)
    stop("nbr.item must be an integer greater or equal to 1")

  if(length(nbr.options)==1) nbr.options <- rep(nbr.options, nbr.item)
  if(min(nbr.options)<2 || !prod(nbr.options==round(nbr.options)))
    stop("nbr.options must be integers greater or equal to 2")

  nbr.option.tot <- sum(nbr.options)

  items.pos <- c(0, cumsum(nbr.options[-nbr.item]))

  if(is.null(items.label)) items.label <- 1:nbr.item
  else if(length(items.label)==1) items.label <- paste(items.label, 1:nbr.item, sep="")
  else if(length(items.label)!=nbr.item) stop("wrong number of item label")

  item <- items.label

  graded <- FALSE
  polytomous <- FALSE
  options.weights <- rep(1, nbr.item)
  mixed <- (length(model) > 1)

  if (mixed) {
    mod.lst <- lapply(1:nbr.item, function(j){
      as.data.frame(genirt(1, model[j], nbr.options[j],
      genslope, genthresh, genasymp, genslopenom, fixed.slope, 
      items.label[j], if(is.list(options.label)){options.label[[j]]}else{options.label}, 
      if(is.vector(key)){key[j]}else{key}))})
    key <- c(lapply(mod.lst, function(mod){k<-attributes(mod)$key;if(is.null(k)) NA else k}), recursive=TRUE)
    item <- rep(item, nbr.options)
    option <- c(lapply(mod.lst, function(mod)attributes(mod)$option), recursive=TRUE)
    options.label <- lapply(mod.lst, function(mod)attributes(mod)$options.label[[1]])
    options.weights <- c(lapply(mod.lst, function(mod)attributes(mod)$options.weights), recursive=TRUE)
    object <- do.call("rbind", mod.lst)
    polytomous <- TRUE

  } else if (model == "1PLM") {
    object <- data.frame(row.names=1:length(item), item=1:nbr.item,
      a=rep(fixed.slope, nbr.item), b=genthresh(nbr.item))
    object$intercept <- -object$a*object$b

  } else if (model == "2PLM") {
    object <- data.frame(row.names=1:length(item), item=1:nbr.item,
      a=genslope(nbr.item), b=genthresh(nbr.item))
    object$intercept <- -object$a*object$b

  } else if (model == "3PLM") {
    object <- data.frame(row.names=1:length(item), item=1:nbr.item,
      a=genslope(nbr.item), b=genthresh(nbr.item), c=genasymp(nbr.item))
    object$intercept <- -object$a*object$b

  } else if (model == "GRADED") {
    item <- rep(item, nbr.options)
    if(is.null(options.label)) {
      options.label <- lapply(nbr.options, function(m)1:m)
    } else if (!is.list(options.label)) {
      options.label <- lapply(nbr.options, function(m)options.label[1:m])
    }
    option <- c(options.label, recursive=TRUE)
    slopes <- genslope(nbr.item)
    bound <- do.call("c", lapply(1:nbr.item,function(j){list(sort(genthresh(nbr.options[j]-1)))}))
    modal <- lapply(1:nbr.item,
      function(j){list((c(bound[[j]][1],bound[[j]])+c(bound[[j]],
	bound[[j]][nbr.options[j]-1]))/2)})
    intercept <- lapply(1:nbr.item,function(j){c(-bound[[j]]*slopes[j],NA)})
    object <- data.frame(row.names=1:length(item), item=item, option=as.character(option),
      a=rep(slopes, nbr.options), b=c(modal, recursive=TRUE),
      intercept=c(intercept, recursive=TRUE))
    polytomous <- TRUE
    graded <- TRUE
    options.weights <- c(sapply(1:nbr.item,function(j)(1:nbr.options[j])),recursive=TRUE)

  } else if (model == "NOMINAL") {
    item <- rep(item, nbr.options)
    if(is.null(options.label)) {
      options.label <- lapply(nbr.options, function(m)intToUtf8(1:m+64,multiple=TRUE))
    } else if (!is.list(options.label)) {
      options.label <- lapply(nbr.options, function(m)options.label[1:m])
    }
    option <- c(options.label, recursive=TRUE)
    slopes <- lapply(1:nbr.item, function(j)genslopenom(nbr.options[j]))
    thresh <- lapply(1:nbr.item, function(j)genthresh(nbr.options[j]))
    intercept <- lapply(1:nbr.item,function(j)-slopes[[j]]*thresh[[j]])
    slopes.mean <- sapply(slopes, mean)
    intercept.mean <- sapply(intercept, mean)
    slopes <- lapply(1:nbr.item,function(j)slopes[[j]]-slopes.mean[j])
    if(is.null(key)) {
      key <- sapply(slopes, which.max)
    } else {
      key <- sapply(1:nbr.item, function(j)match(key[j], options.label[[j]]))
      slopes <- lapply(1:nbr.item,function(j){
        s <- slopes[[j]]
        smax <- which.max(s)
        s[key[j]] <- s[smax]
        s[smax] <- s[key[j]]
        s })
    }
    intercept <- lapply(1:nbr.item,function(j)intercept[[j]]-intercept.mean[j])
    thresh <- lapply(1:nbr.item,function(j)-intercept[[j]]/slopes[[j]])
    object <- data.frame(row.names=1:length(item), item=item, option=option,
      a=c(slopes, recursive=TRUE), b=c(thresh, recursive=TRUE),
      intercept=c(intercept, recursive=TRUE))
    options.weights <- rep(0, nbr.option.tot)
    options.weights[key+items.pos] <- 1
    key <- as.character(object$option[items.pos+key])
    polytomous <- TRUE

  }

  if(!polytomous) {
    nbr.option.tot <- nbr.item
    items.pos <- 0:(nbr.item-1)
    nbr.options <- rep(1, nbr.item)
    option <- rep(1, nbr.item)
  }

  attributes(object)$polytomous <- polytomous
  attributes(object)$graded <- graded
  attributes(object)$mixed <- mixed
  attributes(object)$parametric <- TRUE
  attributes(object)$model <- model
  attributes(object)$key <- key
  attributes(object)$nbr.item <- nbr.item
  attributes(object)$item <- item
  attributes(object)$option <- option
  attributes(object)$nbr.options <- nbr.options
  attributes(object)$nbr.option.tot <- nbr.option.tot
  attributes(object)$items.pos <- items.pos
  attributes(object)$options.label <- options.label
  attributes(object)$call <- match.call()
  attributes(object)$method <- "RANDOM"
  attributes(object)$se <- matrix(NA, nrow=dim(object)[1], ncol=dim(object)[2])
  attributes(object)$options.weights <- options.weights

  class(object) <- c("rirt", class(object))

  return(object)
}

predict.rirt <- function(object, select=NULL, type=NULL,
  z=NULL, data=NULL, normal.ogive=FALSE, z.method="EAP",
  show.se=FALSE, max.nr.iter=100, precision=0.001,
  verbose=0, continue.on.error=1)
{
  parametric <- attributes(object)$parametric
  polytomous <- attributes(object)$polytomous
  graded <- attributes(object)$graded
  mixed <- attributes(object)$mixed
  model <- attributes(object)$model
  key <- attributes(object)$key
  nbr.item <- attributes(object)$nbr.item
  nbr.options <- attributes(object)$nbr.options
  items.pos <- attributes(object)$items.pos
  nbr.option.tot <- attributes(object)$nbr.option.tot
  options.label <- attributes(object)$options.label
  item <- attributes(object)$item
  option <- attributes(object)$option
  method <- attributes(object)$method
  se <- attributes(object)$se 
  options.weights <- attributes(object)$options.weights

  if(is.null(select)) select <- 1:nbr.item
  if(length(select)<1 || min(select)<1 || max(select)>nbr.item
      || !prod(select==round(select)))
    stop("select must be integers from 1 to the number of items")

  if(polytomous) option.select <- do.call("c", lapply(select,
    function(j)items.pos[j]+(1:nbr.options[j])))
  else option.select <- select

  if(parametric) {
    object <- object[option.select,]
    if(!is.null(se)) se <- se[option.select,]
  } else { 
    class.sav <- class(object)
    object <- object[,c(1,option.select+1)]
    class(object) <- class.sav
    if(!is.null(se)) se <- se[,c(1,option.select+1)]
  }

  if(mixed) model <- model[select]
  if(mixed && length(unique(model)) == 1) {
    model <- model[1]
    if(model=="GRADED") {
      graded <- TRUE
    }
  }
  mixed <- (length(model) > 1)
  key <- key[select]
  nbr.item <- length(select)
  nbr.options <- nbr.options[select]
  items.pos <- c(0,cumsum(nbr.options[-nbr.item]))
  nbr.option.tot <- sum(nbr.options)
  options.label <- options.label[select]
  item <- item[option.select]
  option <- option[option.select]
  options.weights <- options.weights[option.select]

  attributes(object)$key <- key
  attributes(object)$nbr.item <- nbr.item
  attributes(object)$nbr.options <- nbr.options
  attributes(object)$items.pos <- items.pos
  attributes(object)$nbr.option.tot <- nbr.option.tot
  attributes(object)$options.label <- options.label
  attributes(object)$item <- item
  attributes(object)$option <- option
  attributes(object)$se <- se
  attributes(object)$parametric <- parametric
  attributes(object)$polytomous <- polytomous
  attributes(object)$graded <- graded
  attributes(object)$model <- model
  attributes(object)$call <- match.call()
  attributes(object)$method <- "SELECT"
  attributes(object)$options.weights <- options.weights

  item.unique <- item[items.pos+1]

  if(!is.null(key))
    key.num <- sapply(1:nbr.item, function(j)
      match(key[j], options.label[[j]]))

  if(is.null(type)) return(object)

  choices <- c("COEFFICIENTS", "OCC", "ICC", "TCC", "BOUNDARY", "INFORMATION",
    "RANDOM_DATA", "LIKELIHOOD", "Z", "FIT_TEST", "LOCAL_INDEPENDENCE_TEST",
    "CTT", "KEY")
  type <- pmatch(toupper(type), choices, duplicates.ok=TRUE)
  if(is.na(type)) stop("invalid type")
  type <- choices[type]

  if(type == "BOUNDARY" && (mixed || model != "GRADED"))
    stop("BOUNDARY is only valid for the GRADED model")

  if(show.se && is.null(se))
    stop("no standard errors in this model")

  .C("set_verbose", as.integer(verbose),
    PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)

  .C("continue_on_error", as.integer(continue.on.error),
    PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)

  if(type == "COEFFICIENTS"){
    if(!parametric) stop("no parameters in nonparametric model")
    if(normal.ogive) {
      object$a <- object$a/1.702
      attributes(object)$se$a.se <- attributes(object)$se$a.se/1.702
      object$intercept <- object$intercept/1.702
    }
    coefs <- as.data.frame(object)
    if(show.se) coefs <- cbind(coefs, se[,-1])
    return(coefs)
  }

  if(type == "KEY") return(key)

  if(is.null(z)) {
    if(type == "RANDOM_DATA") stop("z must be provided to generate random data")
    if(parametric) z <- seq(-4, 4, length=64)
    else {
      z <- object[,1]
      object <- cbind(object[,-1])
    }
  } else {
    if(type %in% c("OCC", "ICC", "TCC", "BOUNDARY", "INFORMATION", "LIKELIHOOD"))
    if(!parametric) {
      if(show.se) stop("no new z allow with show.se")
      object <- apply(object, 2, function(y)
        spline(object[,1], y, xout=z)$y)
      if(is.null(dim(object))) object <- rbind(object[-1])
      else object <- cbind(object[,-1])
    }
  }
  nbr.quad <- length(z)
  quad.limit <- c(-Inf, z, Inf)
  quad.limit <- (quad.limit[-1]+quad.limit[-length(quad.limit)])/2
  quad.weights <- pnorm(quad.limit[-1])-pnorm(quad.limit[-length(quad.limit)])

  if (!polytomous) {

    if(parametric) {
      if (model == "3PLM") asymp <- object$c
      else asymp <- rep(0, nbr.item)
      icc <- do.call("rbind", lapply(z,
        function(z)asymp+(1-asymp)/(1+exp(-object$a*(z-object$b)))))
    } else {
      icc <- as.matrix(object)
    }

    if(type == "ICC") {
      icc <- cbind(z, icc)
      colnames(icc) <- c("z", paste("ICC", item, sep="."))
      if(!parametric && show.se) icc <- cbind(icc, se[,-1])
      return(icc)

    } else if(type == "OCC") {
      occ <- do.call("cbind", lapply(1:nbr.item, function(j)
        cbind(1-icc[,j],icc[,j])))
      occ <- cbind(z, occ)
      colnames(occ) <- c("z", paste("OCC", rep(item, each=2), rep(0:1,nbr.item), sep="."))
      return(occ)

    } else if(type == "TCC") {
      tcc <- apply(icc, 1, sum)
      tcc <- cbind(z, tcc)
      colnames(tcc) <- c("z", "TCC")
      return(tcc)

    } else if(type == "INFORMATION") {
      out <- .C("info_from_probs_wrapper",
        nbr.item=as.integer(nbr.item), 
        nbr.quad=as.integer(nbr.quad), 
        probs=as.double(icc), 
        quad.points=as.double(z),
        info=as.double(rep(0.0, nbr.quad*nbr.item)),
        test.info=as.double(rep(0.0, nbr.quad)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      info <- data.frame(matrix(c(z, out$info), nrow=nbr.quad))
      info <- info[-c(1,nbr.quad),]
      colnames(info) <- c("z", paste("I", item, sep="."))
      return(info)

    } else if(type == "RANDOM_DATA") {
      data <- t(apply(icc, 1, function(p)rbinom(nbr.item, 1, p)))
      nbr.subject <- length(z)
      subject <- names(z)
      if(is.null(subject)) subject <- 1:nbr.subject
      colnames(data) <- item[items.pos+1]
      rownames(data) <- subject
      return(data)

    }

    if(is.null(data)) stop("data is missing")
    if(is.null(dim(data))) data <- matrix(data, nrow=1)
    if(dim(data)[2] != nbr.item) stop("incorrect number of items in data")
    nbr.subject <- dim(data)[1]
    subject <- rownames(data)
    if(is.null(subject)) subject <- 1:nbr.subject
    data[is.na(data)] <- -1

    if(type=="LIKELIHOOD") {
      out <- .C("likelihood_wrapper",
        nbr.subject=as.integer(nbr.subject), nbr.item=as.integer(nbr.item),
        nbr.quad=as.integer(nbr.quad), data=as.integer(t(data)),
        probs=as.double(icc), 
        like=as.double(rep(0.0, nbr.quad*nbr.subject)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      like <- data.frame(matrix(c(z, out$like), nrow=nbr.quad))
      colnames(like) <- c("z", paste("L", subject, sep="."))
      return(like)

    } else if(type=="Z") {
      if(toupper(z.method)=="EAP"){
        out <- .C("eap_abilities_wrapper",
          nbr.subject=as.integer(nbr.subject), nbr.item=as.integer(nbr.item),
          nbr.quad=as.integer(nbr.quad), data=as.integer(t(data)),
          probs=as.double(icc), quad.points=as.double(z),
          quad.weights=as.double(quad.weights),
          abilities=as.double(rep(0.0, nbr.subject)),
          abilities.stddev=as.double(rep(0.0, nbr.subject)),
          PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      } else {
        out <- .C("wmle_abilities_wrapper",
          max.nr.iter=as.integer(max.nr.iter), precision=as.double(precision),  
          nbr.subject=as.integer(nbr.subject), nbr.item=as.integer(nbr.item),
          nbr.quad=as.integer(nbr.quad), data=as.integer(t(data)),
          probs=as.double(icc), quad.points=as.double(z),
          abilities=as.double(rep(0.0, nbr.subject)),
          abilities.stddev=as.double(rep(0.0, nbr.subject)),
          PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      }
      z <- data.frame(Z=out$abilities)
      if(show.se) z$se <- out$abilities.stddev
      rownames(z) <- subject
      return(z)

    } else if(type=="FIT_TEST") {
      out <- .C("llk_ratio_fit_test_wrapper",
        nbr.subject=as.integer(nbr.subject), 
        nbr.item=as.integer(nbr.item), 
        nbr.quad=as.integer(nbr.quad), 
        data=as.integer(t(data)),
        quad.weights=as.double(quad.weights),
        probs=as.double(icc),
        nbr.inter=as.integer(11),
        chi2=as.double(rep(0.0, nbr.item+1)),
        df=as.integer(rep(0, nbr.item+1)),
        p.value=as.double(rep(0.0, nbr.item+1)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      test <- data.frame(chi2=out$chi2, df=out$df, p.value=out$p.value)
      rownames(test) <- c(item, "TOTAL")
      return(test)

    } else if(type=="LOCAL_INDEPENDENCE_TEST") {
      out <- .C("llk_ratio_ld_test_wrapper",
        nbr.subject=as.integer(nbr.subject), 
        nbr.item=as.integer(nbr.item), 
        nbr.quad=as.integer(nbr.quad), 
        data=as.integer(t(data)),
        probs=as.double(icc),
        quad.weights=as.double(quad.weights),
        chi2=as.double(rep(0.0, nbr.item*nbr.item)),
        df=as.integer(rep(0, nbr.item*nbr.item)),
        p.value=as.double(rep(0.0,nbr.item*nbr.item)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      test <- data.frame(item1=rep(item, each=nbr.item), item2=rep(item, nbr.item),
        chi2=c(out$chi2), df=c(out$df), p.value=c(out$p.value))
      return(test)

    } else if(type=="CTT") {
      out <- .C("classical_statistics_wrapper",
        nbr.subject=as.integer(nbr.subject), 
        nbr.item=as.integer(nbr.item), 
        data=as.integer(t(data)),
        items.mean=as.double(rep(0,nbr.item)),
        items.sd=as.double(rep(0,nbr.item)),
        items.corr=as.double(rep(0,nbr.item)),
        items.poly.corr=as.double(rep(0,nbr.item)),
        items.nbr=as.integer(rep(0,nbr.item)),
        subjects.score=as.double(rep(0,nbr.subject)),
        subjects.nbr=as.integer(rep(0,nbr.subject)),
        nbr=as.integer(0), mean=as.double(0), 
        sd=as.double(0), alpha=as.double(0),
        pairs.corr=as.double(rep(0,nbr.item*nbr.item)),
        freq.table=as.integer(rep(0,nbr.item*3)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      out$freq.table <- lapply(1:nbr.item, function(j){
          item.freq <- out$freq.table[(j-1)*3+(1:3)]
          names(item.freq) <- c("NA", "0", "1")
          item.freq})
      ctt <- list(
        items=data.frame(item=item,
          mean=out$items.mean, sd=out$items.sd,
          corr=out$items.corr, poly.cor=out$items.poly.corr,
          nbr=out$items.nbr),
        subjects=data.frame(subject=subject,
          score=out$subjects.score, nbr=out$subjects.nbr),
        nbr=out$nbr, mean=out$mean, sd=out$sd, alpha=out$alpha, 
        pairs.corr=matrix(out$pairs.corr, nrow=nbr.item),
        freq.table=out$freq.table)
      class(ctt) <- c("rirtctt", class(ctt))
      return(ctt)

    } else stop("invalid type")

  } else { # polytomous

    if(!parametric) {
      occ <- as.matrix(object)

    } else if (mixed) {
      occ <- do.call("cbind", lapply(1:nbr.item, function(j)
        predict(object, j, "occ", z=z)[,-1]))

    } else if (graded) {
      slopes <- object$a[items.pos+1]
      intercept <- object$intercept[!is.na(object$intercept)]
      item.bound <- item[!is.na(object$intercept)]
      bound <- do.call("cbind", lapply(1:nbr.item, function(j){
        t(sapply(z, function(z)
          c(1,1/(1+exp(-slopes[j]*z-intercept[item.bound==item.unique[j]]))))) }))

      if(type == "BOUNDARY") {
        bound <- cbind(z, bound)
        colnames(bound) <- c("z", paste("BOUND", item, option, sep="."))
	bound <- bound[,-(items.pos+2)]
        return(bound)
      }

      occ <- do.call("cbind", lapply(item.unique, function(j)
        t(sapply(1:nbr.quad, function(i)-diff(c(bound[i,item==j],0))))))

    } else { # nominal
      occ <- do.call("cbind", lapply(item.unique, function(j) {
        t(sapply(z, function(z) {
          p <- exp(object$a[item==j]*(z-object$b[item==j]))
          p/sum(p) })) }))
    }

    if(type == "OCC") {
      occ <- cbind(z, occ)
      colnames(occ) <- c("z", paste("OCC", item, option, sep="."))
      if(!parametric && show.se) occ <- cbind(occ, se[,-1])
      return(occ)

    } else if(type == "ICC" || type == "TCC") {
      if(graded || is.null(key)) icc <- do.call("cbind", lapply(1:nbr.item,
        function(j)occ[,item==item.unique[j]] %*% options.weights[item==item.unique[j]]))
      else icc <- occ[,items.pos+key.num]

      if(type == "TCC") {
        tcc <- cbind(z, apply(icc, 1, sum))
        colnames(tcc) <- c("z", "TCC")
        return(tcc)

      } else { # icc
        icc <- cbind(z, icc)
        colnames(icc) <- c("z", paste("ICC", item.unique, sep="."))
        return(icc)
      }

    } else if(type=="INFORMATION") {
      out <- .C("info_from_probs_mc_wrapper",
        nbr.item=as.integer(nbr.item),
        nbr.option.tot=as.integer(nbr.option.tot),
        nbr.quad=as.integer(nbr.quad), 
        probs=as.double(occ), quad.points=as.double(z),
        nbr.options=as.integer(nbr.options),
        items.pos=as.integer(items.pos),
        option.info=as.double(rep(0.0, nbr.quad*nbr.option.tot)),
        info=as.double(rep(0.0, nbr.quad*nbr.item)),
        test.info=as.double(rep(0.0, nbr.quad)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      info <- data.frame(matrix(c(z, out$info), nrow=nbr.quad))
      info <- info[-c(1,nbr.quad),]
      colnames(info) <- c("z", paste("I", item.unique, sep="."))
      return(info)

    } else if(type == "RANDOM_DATA") {
      data <- t(apply(occ, 1, function(p)
        sapply(1:nbr.item, function(j)
          options.label[[j]][which.max(rmultinom(1, 1, 
            p[items.pos[j]+(1:nbr.options[j])]))])))
      nbr.subject <- length(z)
      subject <- names(z)
      if(is.null(subject)) subject <- 1:nbr.subject
      colnames(data) <- item[items.pos+1]
      rownames(data) <- subject
      return(data)
    }

    if(is.null(data)) stop("data is missing")
    if(is.null(dim(data))) data <- matrix(data, nrow=1)
    if(dim(data)[2] != nbr.item) stop("incorrect number of items in data")
    nbr.subject <- dim(data)[1]
    subject <- rownames(data)
    if(is.null(subject)) subject <- 1:nbr.subject

    data.num <- sapply(1:nbr.item, function(j)match(as.character(data[,j]), options.label[[j]]))
    if(is.null(dim(data.num))) data.num <- matrix(data.num, nrow=1)
    data.exp <- t(apply(data.num, 1, function(x){
      y<-rep(0, nbr.option.tot)
      y[x+items.pos] <- 1
      y }))
    data.num[is.na(data.num)] <- -1
    data.exp[is.na(data.exp)] <- -1

    if(type == "LIKELIHOOD") {
      out <- .C("likelihood_mc_wrapper",
        nbr.subject=as.integer(nbr.subject), nbr.item=as.integer(nbr.item),
        nbr.quad=as.integer(nbr.quad), nbr.option.tot=as.integer(nbr.option.tot),
        data=as.integer(t(data.exp)), probs=as.double(occ), 
        nbr.options=as.integer(nbr.options), items.pos=as.integer(items.pos),
        like=as.double(rep(0.0, nbr.quad*nbr.subject)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      like <- data.frame(matrix(c(z, out$like), nrow=nbr.quad))
      colnames(like) <- c("z", paste("L", subject, sep="."))
      return(like)

    } else if(type=="Z") {
      if(toupper(z.method)=="EAP"){
        out <- .C("eap_abilities_mc_wrapper",
          nbr.subject=as.integer(nbr.subject), nbr.item=as.integer(nbr.item),
          nbr.quad=as.integer(nbr.quad), nbr.option.tot=as.integer(nbr.option.tot),
          items.pos=as.integer(items.pos), nbr.options=as.integer(nbr.options), 
          data=as.integer(t(data.exp)), probs=as.double(occ), quad.points=as.double(z),
          quad.weights=as.double(quad.weights),
          abilities=as.double(rep(0.0, nbr.subject)),
          abilities.stddev=as.double(rep(0.0, nbr.subject)),
          PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      } else {
        out <- .C("wmle_abilities_mc_wrapper",
          max.nr.iter=as.integer(max.nr.iter), precision=as.double(precision),  
          nbr.subject=as.integer(nbr.subject), nbr.item=as.integer(nbr.item),
          nbr.quad=as.integer(nbr.quad), nbr.option.tot=as.integer(nbr.option.tot),
          nbr.options=as.integer(nbr.options), items.pos=as.integer(items.pos),
          data=as.integer(t(data.exp)), probs=as.double(occ), quad.points=as.double(z),
          abilities=as.double(rep(0.0, nbr.subject)),
          abilities.stddev=as.double(rep(0.0, nbr.subject)),
          PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      }
      z <- data.frame(Z=out$abilities)
      if(show.se) z$se <- out$abilities.stddev
      rownames(z) <- subject
      return(z)

    } else if(type=="FIT_TEST") {
      out <- .C("llk_ratio_fit_test_mc_wrapper",
        nbr.subject=as.integer(nbr.subject), 
        nbr.item=as.integer(nbr.item), 
        nbr.quad=as.integer(nbr.quad), 
        nbr.option.tot=as.integer(nbr.option.tot),
        data=as.integer(t(data.exp)),
        quad.weights=as.double(quad.weights),
        probs=as.double(occ),
        nbr.options=as.integer(nbr.options),
        items.pos=as.integer(items.pos),
        nbr.inter=as.integer(11),
        chi2=as.double(rep(0.0, nbr.item+1)),
        df=as.integer(rep(0, nbr.item+1)),
        p.value=as.double(rep(0.0, nbr.item+1)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      test <- data.frame(chi2=out$chi2, df=out$df, p.value=out$p.value)
      rownames(test) <- c(item.unique, "TOTAL")
      return(test)

    } else if(type=="LOCAL_INDEPENDENCE_TEST") {
      out <- .C("llk_ratio_ld_test_mc_wrapper",
        nbr.subject=as.integer(nbr.subject), 
        nbr.item=as.integer(nbr.item), 
        nbr.quad=as.integer(nbr.quad), 
        nbr.option.tot=as.integer(nbr.option.tot),
        data=as.integer(t(data.num)),
        probs=as.double(occ),
        quad.weights=as.double(quad.weights),
        nbr.options=as.integer(nbr.options),
        items.pos=as.integer(items.pos),
        chi2=as.double(rep(0.0, nbr.item*nbr.item)),
        df=as.integer(rep(0, nbr.item*nbr.item)),
        p.value=as.double(rep(0.0, nbr.item*nbr.item)),
        PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      test <- data.frame(item1=rep(item.unique, each=nbr.item),
        item2=rep(item.unique, nbr.item),
        chi2=c(out$chi2), df=c(out$df), p.value=c(out$p.value))
      return(test)

    } else if(type=="CTT") {
      out <- .C("classical_statistics_mc_wrapper",
                nbr.option.tot=as.integer(nbr.option.tot), 
                nbr.subject=as.integer(nbr.subject), 
                nbr.item=as.integer(nbr.item), 
                data=as.integer(t(data.num)),
                options.weights=as.double(options.weights),
                items.pos=as.integer(items.pos), 
                nbr.options=as.integer(nbr.options),
                items.mean=as.double(rep(0,nbr.item)),
                items.sd=as.double(rep(0,nbr.item)),
                items.corr=as.double(rep(0,nbr.item)),
                items.poly.corr=as.double(rep(0,nbr.item)),
                items.nbr=as.integer(rep(0,nbr.item)),
                subjects.score=as.double(rep(0,nbr.subject)),
                subjects.nbr=as.integer(rep(0,nbr.subject)),
                nbr=as.integer(0), mean=as.double(0), 
                sd=as.double(0), alpha=as.double(0),
                pairs.corr=as.double(rep(0,nbr.item*nbr.item)),
                freq.table=as.integer(rep(0,nbr.item+nbr.option.tot)),
                PACKAGE="rirt", NAOK=TRUE, DUP=FALSE)
      out$freq.table <- lapply(1:nbr.item, function(j){
          item.freq <- out$freq.table[items.pos[j]+j+(0:(nbr.options[j]))]
          names(item.freq) <- c("NA", options.label[[j]])
          item.freq})
      ctt <- list(
        items=data.frame(item=item.unique,
          mean=out$items.mean, sd=out$items.sd,
          corr=out$items.corr, poly.corr=out$items.poly.corr,
          nbr=out$items.nbr),
        subjects=data.frame(subject=subject,
          score=out$subjects.score, nbr=out$subjects.nbr),
        nbr=out$nbr, mean=out$mean, sd=out$sd, alpha=out$alpha,
        pairs.corr=matrix(out$pairs.corr, nrow=nbr.item),
        freq.table=out$freq.table)
      class(ctt) <- c("rirtctt", class(ctt))
      return(ctt)

    } else stop("invalid type")

  }

}

summary.rirt <- function (object)
{
  # the summary is the attributes of the object
  summ <- attributes(object)
  class(summ) <- "summary.rirt"
  summ
}

print.summary.rirt <- function(object)
{
  if(!is.null(object$call)) {
  cat("Call:               ")
  print(object$call)
  cat("Number of items:   ", object$nbr.item, "\n")
  if(object$polytomous){
  cat("Numbers of options:", object$nbr.options, "\n")
  }
  if(object$parametric){
  cat("Model:             ", object$model, "\n")
  }
  if(object$model=="NOMINAL" || length(object$model) > 1)
  cat("Key:               ", object$key, "\n")
  if(!(object$method %in% c("RANDOM", "SELECT"))){
  cat("Number of subjects:", object$nbr.subject, "\n")
  cat("Estimation method: ", object$method, "\n")
  if(!object$parametric)
  cat("Smoothing factor:  ", object$smooth.factor, "\n")
  if(object$thresh.prior)
  cat("Threshold's prior: ", "Normal(mean=", object$thresh.mean, 
    ", sd=", object$thresh.dev, ")\n")
  if(object$slope.prior)
  cat("Slope's prior:     ", "LogNormal(mean=", object$slope.mean, 
    ", sd=", object$slope.dev, ")\n")
  if(object$asymp.prior)
  cat("Asymptote's prior: ", "Beta(mean=", object$asymp.mean, 
    ", weight=", object$asymp.weight, ")\n")
  }
  cat("\n")
  }
  invisible(object)
}

print.rirt <- function(object)
{
  print(summary(object))
  print.data.frame(object)
}

plot.rirt <- function(object, select=NULL, type=NULL,
  z=NULL, data=NULL, col=NULL, lty=NULL, pch=NULL, lwd=2,
  xlim=NULL, ylim=NULL,  xlab=NULL, ylab=NULL, main=NULL,
  leg=TRUE, leg.loc="bottomright", leg.title=NULL, leg.lab=NULL,
  leg.width=11, plot.type="l", add=FALSE, ...)
{
  if(is.null(type)) type <- "ICC"
  choices <- c("OCC", "ICC", "TCC", "BOUNDARY", "INFORMATION", "LIKELIHOOD")
  type <- pmatch(toupper(type), choices, duplicates.ok=TRUE)
  if(is.na(type)) stop("invalid type")
  type <- choices[type]

  curves <- predict(object, type=type, select=select, z=z, data=data)
  x <- curves[,1]
  y <- curves[,-1]
  nbr.curve <- dim(y)[2]
  if(is.null(nbr.curve)) nbr.curve <- 1

  if(is.null(xlab)) xlab <- colnames(curves)[1]
  if(is.null(ylab)) ylab <- type

  if(is.null(main)) main <- type
  if(is.null(leg.title)) leg.title <- type
  if(is.null(leg.lab)) leg.lab <- colnames(curves)[-1]

  if(is.null(col)){
    if(nbr.curve < 3) col <- 1:nbr.curve
    else if(nbr.curve < 4) col <- c(1,2,4)
    else if(nbr.curve < 5) col <- c(1,2,4,3)
    else col <- c(1,2,4,3,5:nbr.curve)
  }
  if(is.null(lty)) lty <- 1:nbr.curve
  if(is.null(pch)) pch <- 1:nbr.curve
  pt.lwd <- lwd
  if(plot.type=="p") lty <- NULL
  if(plot.type=="l") pch <- NULL

  if(!add) {
    if(is.null(xlim)) {
      xlim <- range(x, na.rm=TRUE)
    }
    if(is.null(ylim)) {
      ylim <- range(y, na.rm=TRUE)
    }
    plot.new()
    plot.window(xlim=xlim, ylim=ylim)
    if(leg) {
      width <- diff(xlim)
      width <- max(width, width*width/(width-leg.width*strwidth("A")))
      xlim[2] <- xlim[1]+width
    }
    par(new=TRUE)
  }

  matplot(x, y, type=plot.type, col=col, lty=lty, pch=pch, lwd=lwd,
    xlim=xlim, ylim=ylim, xlab=xlab, ylab=ylab, main=main, add=add, ...)

  if(leg) {
    if(plot.type=="p") legend(leg.loc, legend=leg.lab, xjust=1, yjust=0,
      col=col, pt.lwd=lwd, pch=pch, title=leg.title)
    else legend(leg.loc, legend=leg.lab, xjust=1, yjust=0,
      col=col, lty=lty, lwd=lwd, pch=pch, title=leg.title)
  }
}

irt <- function(data, nbr.quad=64, quad.from=-4.0, quad.to=4.0,
  model=2, method="BME", smooth.factor=NULL,
  slope.prior=TRUE, thresh.prior=FALSE, asymp.prior=TRUE,
  slope.mean=1.702, slope.dev=1.649*1.702, thresh.mean=0, thresh.dev=2.0,
  asymp.mean=0.2, asymp.weight=20.0,
  slope.init=slope.mean, thresh.init=thresh.mean, asymp.init=asymp.mean,
  max.em.iter=100, max.nr.iter=100, precision=1e-5,
  grouping=TRUE, verbose=0, probs=NULL)
{
  warning("irt is deprecated, use fitirt instead")
  fitirt(data,
    model=if(method %in% c("BME", "MMLE")){paste(model,"PLM",sep="")}else{method},
    max.em.iter=max.em.iter, max.nr.iter=max.nr.iter, precision=precision,
    smooth.factor=smooth.factor, z=seq(quad.from, quad.to, length=nbr.quad),
    grouping=grouping, verbose=verbose, init=if(!is.null(probs))
    {cbind(seq(quad.from, quad.to, length=nbr.quad),probs)}else{NULL},
    slope.init=slope.init, thresh.init=thresh.init, asymp.init=asymp.init,
    slope.prior=if(slope.prior){c(slope.mean,slope.dev)}else{NULL},
    thresh.prior=if(thresh.prior){c(thresh.mean,thresh.dev)}else{NULL},
    asymp.prior=if(asymp.prior){c(asymp.mean,asymp.weight)}else{NULL})
}

mirt <- function(data, options.weights=NULL, graded=FALSE, key=NULL,
  nbr.quad=64, quad.from=-4.0, quad.to=4.0,
  method="MMLE", smooth.factor=NULL, slope.init=1.702, thresh.init=1.0,
  max.em.iter=100, max.nr.iter=200, precision=1e-5,
  grouping=TRUE, verbose=0, probs=NULL, slopes=NULL, thresh=NULL)
{
  warning("mirt is deprecated, use fitirt instead")
  fitirt(data,
    model=if(method == "MMLE"){if(graded)"GRADED"else"NOMINAL"}else{method},
    graded=graded, key=key,
    max.em.iter=max.em.iter, max.nr.iter=max.nr.iter, precision=precision,
    smooth.factor=smooth.factor, z=seq(quad.from, quad.to, length=nbr.quad),
    grouping=grouping, verbose=verbose, init=if(!is.null(probs))
    {cbind(seq(quad.from, quad.to, length=nbr.quad),probs)}
    else if(!is.null(slopes)){data.frame(a=slopes,b=thresh)}else{NULL},
    slope.init=slope.init, thresh.init=thresh.init)
}

summary.rirtctt <- function (object)
{
  # the summary is the object
  class(object) <- "summary.rirtctt"
  object
}

print.summary.rirtctt <- function(object)
{
  cat("Number of subjects:    ", dim(object$subjects)[1], "\n")
  cat("Number of items:       ", dim(object$items)[1], "\n")
  cat("Number of responses:   ", object$nbr, "\n")
  cat("Score's mean:          ", object$mean, "\n")
  cat("Score's std. dev.:     ", object$sd, "\n")
  cat("Cronbach's alpha:      ", object$alpha, "\n")
  cat("\n")
  invisible(object)
}

print.rirtctt <- function(object, items=TRUE, subjects=FALSE, cor=FALSE)
{
  print(summary(object))
  if (items) print.data.frame(object$items)
  if (subjects) print.data.frame(object$subjects)
  if (cor) print(object$pairs.corr)
}
