/* eap_ability_3plm.c
 *
 * Copyright (C) 2005 Stephane Germain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
   \file
   \brief Functions to estimate the abilities by EAP.
   \author Stephane Germain <germste@gmail.com>
*/

#include "libirt.h"

#include <stdio.h>
#include <math.h>

/**
   \brief Estimate the abilities by Bayes exact a posteriori.

   Work also for expanded multiple choices patterns.

   @param[in] post A matrix(patterns x classes) with the probability that a pattern
   is in a quadrature class.
   @param[in] quad_points A vector(classes) with the middle points of each quadrature class.
   @param[in] quad_weights A vector(classes) with the prior weights of each quadrature class.
   @param[out] abilities A vector(patterns) with the estimated abilities.
   @param[out] abilities_stddev A vector(items) with the standard errors
   of the estimated abilities.

   \warning The memory for \em abilities and \em abilities_stddev should be allocated before.
*/
void
eap_abilities (gsl_matrix * post,
	       gsl_vector * quad_points, gsl_vector * quad_weights,
	       gsl_vector * abilities, gsl_vector * abilities_stddev)
{
  int nbr_quad, nbr_pattern, j, k;
  double mean, var, tmp;

  nbr_quad = quad_points->size;
  nbr_pattern = post->size1;

  /* for each pattern */
  for (j = 0; j < nbr_pattern; j++)
    {
      /* compute the expected ability */
      mean = 0;
      /* sum over the classes */
      for (k = 0; k < nbr_quad; k++)
	{
	  mean += gsl_vector_get (quad_points, k) * gsl_matrix_get (post, j, k);
	}
      gsl_vector_set (abilities, j, mean);

      if (abilities_stddev)
	{
	  /* Compute the standard errors of the abilities. */
	  var = 0;
	  /* sum over the classes */
	  for (k = 0; k < nbr_quad; k++)
	    {
	      tmp =
		gsl_vector_get (quad_points, k) - gsl_vector_get (abilities,
								  j);
	      tmp *= tmp;
	      var +=
		tmp * gsl_vector_get (quad_weights, k) * gsl_matrix_get (post,
									 j,
									 k);
	    }
	  gsl_vector_set (abilities_stddev, j, sqrt (var));
	}
    }
}
